
// chat_module/assets/js/chat.js

(function () {
  const API_BASE = window.CHAT_API_BASE || '../api/chat';
  const chatListEl = document.getElementById('chatList');
  const chatEmptyEl = document.getElementById('chatEmpty');
  const chatWindowEl = document.getElementById('chatWindow');
  const chatHeaderEl = document.getElementById('chatHeader');
  const chatMessagesEl = document.getElementById('chatMessages');
  const chatInputEl = document.getElementById('chatInput');
  const chatSendBtn = document.getElementById('chatSendBtn');
  const tabs = document.querySelectorAll('.tg-tab');

  let currentConversationId = null;
  let currentOtherUser = null;
  let lastMessageId = 0;
  let pollTimer = null;

  function escapeHtml(str) {
    return String(str || '').replace(/[&<>"']/g, function (c) {
      return ({'&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;'}[c]);
    });
  }

  function api(url, options) {
    return fetch(url, options).then(r => r.json());
  }

  function loadChatList(filter) {
    api(API_BASE + '/chat_list.php?filter=' + encodeURIComponent(filter || 'all'))
      .then(data => {
        chatListEl.innerHTML = '';
        (data.conversations || []).forEach(conv => {
          const item = document.createElement('div');
          item.className = 'tg-chat-item';
          item.dataset.cid = conv.conversation_id;
          item.dataset.otherId = conv.other_user_id;
          item.dataset.name = conv.name;

          const avatar = document.createElement('div');
          avatar.className = 'tg-avatar';
          if (conv.avatar) {
            const img = document.createElement('img');
            img.src = conv.avatar;
            avatar.appendChild(img);
          } else {
            avatar.textContent = conv.name ? conv.name.charAt(0).toUpperCase() : '?';
          }

          const main = document.createElement('div');
          main.style.flex = '1';

          const row1 = document.createElement('div');
          row1.style.display = 'flex';
          row1.style.justifyContent = 'space-between';
          const nameEl = document.createElement('div');
          nameEl.textContent = conv.name;
          nameEl.style.fontWeight = '600';
          const timeEl = document.createElement('div');
          timeEl.style.fontSize = '11px';
          timeEl.style.color = '#7a8696';
          timeEl.textContent = conv.last_time ? conv.last_time.substring(0, 16) : '';
          row1.appendChild(nameEl);
          row1.appendChild(timeEl);

          const row2 = document.createElement('div');
          row2.style.display = 'flex';
          row2.style.justifyContent = 'space-between';
          const previewEl = document.createElement('div');
          previewEl.style.fontSize = '13px';
          previewEl.style.color = '#7a8696';
          previewEl.textContent = conv.last_message || '';
          const unreadEl = document.createElement('div');
          if (conv.unread > 0) {
            unreadEl.textContent = conv.unread;
            unreadEl.style.minWidth = '18px';
            unreadEl.style.height = '18px';
            unreadEl.style.borderRadius = '999px';
            unreadEl.style.background = '#2b77c5';
            unreadEl.style.color = '#fff';
            unreadEl.style.fontSize = '11px';
            unreadEl.style.display = 'flex';
            unreadEl.style.alignItems = 'center';
            unreadEl.style.justifyContent = 'center';
          }
          row2.appendChild(previewEl);
          row2.appendChild(unreadEl);

          main.appendChild(row1);
          main.appendChild(row2);

          item.appendChild(avatar);
          item.appendChild(main);

          item.addEventListener('click', () => openConversation(conv));
          chatListEl.appendChild(item);
        });
      })
      .catch(console.error);
  }

  function openConversation(conv) {
    currentConversationId = conv.conversation_id;
    currentOtherUser = conv.other_user_id;
    lastMessageId = 0;

    Array.from(document.querySelectorAll('.tg-chat-item')).forEach(el => {
      el.classList.toggle('tg-chat-item-active', el.dataset.cid == currentConversationId);
    });

    chatEmptyEl.classList.add('hidden');
    chatWindowEl.classList.remove('hidden');

    chatHeaderEl.innerHTML = `
      <div style="display:flex;align-items:center;">
        <div class="tg-avatar" style="margin-right:10px;">${escapeHtml(conv.name.charAt(0))}</div>
        <div>
          <div class="tg-chat-header-title">${escapeHtml(conv.name)}</div>
          <div class="tg-chat-header-sub" id="chatStatus">online</div>
        </div>
      </div>
    `;

    chatMessagesEl.innerHTML = '';
    loadMessages(true);

    if (pollTimer) clearInterval(pollTimer);
    pollTimer = setInterval(() => {
      loadMessages(false);
      refreshPresence();
    }, 1500);
  }

  function loadMessages(initial) {
    if (!currentConversationId) return;

    api(API_BASE + '/fetch_messages.php?conversation_id=' + encodeURIComponent(currentConversationId) +
        '&since_id=' + encodeURIComponent(lastMessageId))
      .then(data => {
        let maxId = lastMessageId;
        (data.messages || []).forEach(msg => {
          maxId = Math.max(maxId, parseInt(msg.id, 10));
          const bubble = document.createElement('div');
          const mine = parseInt(msg.sender_id, 10) === window.LOGGED_IN_USER_ID;
          bubble.className = 'tg-bubble ' + (mine ? 'tg-bubble-me' : 'tg-bubble-them');
          bubble.innerHTML = `
            <div>${escapeHtml(msg.body)}</div>
            <div class="tg-bubble-meta">${msg.created_at.substring(11,16)}</div>
          `;
          chatMessagesEl.appendChild(bubble);
        });

        if (maxId > lastMessageId) {
          lastMessageId = maxId;
          chatMessagesEl.scrollTop = chatMessagesEl.scrollHeight;
          api(API_BASE + '/mark_seen.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'conversation_id=' + encodeURIComponent(currentConversationId) +
                  '&last_id=' + encodeURIComponent(lastMessageId)
          });
        }
      })
      .catch(console.error);
  }

  function sendMessage() {
    const text = chatInputEl.value.trim();
    if (!text || !currentConversationId) return;

    api(API_BASE + '/send_message.php', {
      method: 'POST',
      headers: {'Content-Type': 'application/x-www-form-urlencoded'},
      body: 'conversation_id=' + encodeURIComponent(currentConversationId) +
            '&body=' + encodeURIComponent(text)
    }).then(data => {
      if (data && data.ok) {
        chatInputEl.value = '';
        loadMessages(false);
        loadChatList('all');
      }
    }).catch(console.error);
  }

  function refreshPresence() {
    if (!currentOtherUser) return;
    api(API_BASE + '/presence.php?other_user_id=' + encodeURIComponent(currentOtherUser))
      .then(data => {
        const el = document.getElementById('chatStatus');
        if (!el) return;
        if (data.typing) {
          el.textContent = 'typing…';
        } else if (data.online) {
          el.textContent = 'online';
        } else if (data.last_active) {
          el.textContent = 'last seen ' + data.last_active.substring(11,16);
        } else {
          el.textContent = '';
        }
      }).catch(console.error);
  }

  chatSendBtn && chatSendBtn.addEventListener('click', sendMessage);
  chatInputEl && chatInputEl.addEventListener('keydown', function (e) {
    if (e.key === 'Enter') {
      e.preventDefault();
      sendMessage();
    }
  });

  tabs.forEach(tab => {
    tab.addEventListener('click', () => {
      tabs.forEach(t => t.classList.remove('tg-tab-active'));
      tab.classList.add('tg-tab-active');
      loadChatList(tab.dataset.filter);
    });
  });

  loadChatList('all');
})();
